import React, { useState, useEffect } from 'react';
import { reportService } from '../services/phase4Services';
import {
    BarChart3,
    ArrowUpRight,
    ArrowDownRight,
    Wallet,
    Download,
    Calendar,
    Filter,
    TrendingUp,
    TrendingDown,
    Activity,
    RefreshCw
} from 'lucide-react';
import { clsx } from 'clsx';
import settingsService from '../services/settingsService';

const ProfitLossReport = () => {
    const [report, setReport] = useState(null);
    const [loading, setLoading] = useState(true);
    const [currency, setCurrency] = useState('KES');

    useEffect(() => {
        fetchReport();
        settingsService.getSettings().then(res => setCurrency(res.data.DEFAULT_CURRENCY || 'KES'));
    }, []);

    const fetchReport = async () => {
        setLoading(true);
        try {
            const res = await reportService.getProfitLoss();
            setReport(res.data);
        } catch (err) {
            console.error(err);
        } finally {
            setLoading(false);
        }
    };

    if (loading) return (
        <div className="min-h-[60vh] flex flex-col items-center justify-center gap-4 animate-pulse">
            <div className="w-12 h-12 border-4 border-primary/20 border-t-primary rounded-full animate-spin"></div>
            <p className="text-[10px] font-black text-slate-400 uppercase tracking-[0.2em]">Loading report...</p>
        </div>
    );

    return (
        <div className="space-y-10 animate-slide-up pb-20">
            {/* Premium Header */}
            <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-6">
                <div className="flex items-center gap-4 sm:gap-5 min-w-0">
                    <div className="p-3 sm:p-4 bg-emerald-500 rounded-2xl text-white shadow-xl shadow-emerald-500/20 shrink-0">
                        <Activity className="w-6 h-6 sm:w-8 sm:h-8 shrink-0" />
                    </div>
                    <div className="min-w-0">
                        <h1 className="text-xl sm:text-2xl font-extrabold text-slate-800 dark:text-white tracking-tight leading-none truncate">Profit & Loss Statement</h1>
                        <div className="flex items-center gap-2 mt-1">
                            <span className="text-[10px] font-black text-slate-400 dark:text-slate-500 uppercase tracking-widest whitespace-nowrap">P & L Analysis</span>
                            <div className="w-1 h-1 bg-slate-300 dark:bg-slate-700 rounded-full shrink-0"></div>
                            <span className="text-[10px] font-black text-emerald-500 uppercase tracking-widest whitespace-nowrap">Summary</span>
                        </div>
                    </div>
                </div>
                <div className="flex items-center gap-3 w-full md:w-auto">
                    <button
                        onClick={fetchReport}
                        className="btn-secondary h-12 w-12 sm:h-14 sm:w-14 flex items-center justify-center rounded-2xl shrink-0"
                        title="Recalculate Statement"
                    >
                        <RefreshCw className="w-4 h-4 shrink-0" />
                    </button>
                    <button className="flex-1 md:flex-none btn-primary flex items-center justify-center gap-2 px-8 py-3 h-12 lg:h-14 shadow-2xl">
                        <Download className="w-4 h-4 shrink-0" />
                        <span className="text-[10px] sm:text-[11px] font-black uppercase tracking-widest whitespace-nowrap">Export Statement</span>
                    </button>
                </div>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-4 gap-6 sm:gap-8">
                <div className="premium-card p-6 sm:p-10 col-span-1 md:col-span-2 relative overflow-hidden group">
                    <div className={clsx(
                        "absolute top-0 right-0 w-48 h-48 -mr-24 -mt-24 rounded-full blur-3xl transition-all duration-700",
                        report.netProfit >= 0 ? "bg-emerald-500/10 group-hover:bg-emerald-500/20" : "bg-rose-500/10 group-hover:bg-rose-500/20"
                    )} />

                    <div className="relative">
                        <p className="text-[10px] font-black text-slate-400 dark:text-slate-500 uppercase tracking-[0.2em] mb-4">Net Profit</p>
                        <div className="flex flex-col sm:flex-row sm:items-baseline gap-4">
                            <h3 className={clsx(
                                "text-4xl sm:text-5xl font-black tracking-tighter",
                                report.netProfit >= 0 ? "text-emerald-500" : "text-rose-500 font-bold"
                            )}>
                                <span className="text-xl opacity-40 mr-2">{currency}</span>
                                {Number(report.netProfit).toLocaleString()}
                            </h3>
                            <div className={clsx(
                                "inline-flex items-center gap-1 self-start px-3 py-1 rounded-full text-[9px] font-black uppercase tracking-widest border",
                                report.netProfit >= 0 ? "bg-emerald-500/10 text-emerald-500 border-emerald-500/20" : "bg-rose-500/10 text-rose-500 border-rose-500/20"
                            )}>
                                {report.netProfit >= 0 ? <TrendingUp className="w-3 h-3" /> : <TrendingDown className="w-3 h-3" />}
                                {report.netProfit >= 0 ? 'Profit' : 'Loss'}
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div className="max-w-4xl mx-auto">
                <div className="premium-card overflow-hidden">
                    <div className="p-6 sm:p-8 border-b border-slate-100 dark:border-slate-800/50 bg-slate-50/50 dark:bg-slate-900/40 flex justify-between items-center">
                        <div className="flex items-center gap-3">
                            <BarChart3 className="w-5 h-5 text-primary shrink-0" />
                            <h2 className="text-[11px] font-black text-slate-700 dark:text-slate-300 uppercase tracking-widest">Breakdown</h2>
                        </div>
                    </div>

                    <div className="p-6 sm:p-10 space-y-12 sm:space-y-16">
                        {/* Income Section */}
                        <div className="space-y-6 sm:space-y-8">
                            <div className="flex flex-col sm:flex-row sm:justify-between sm:items-center gap-4 pb-4 border-b border-slate-900/10 dark:border-white/10">
                                <div className="flex items-center gap-4">
                                    <div className="p-3 bg-emerald-500/10 rounded-xl text-emerald-500 shrink-0">
                                        <TrendingUp className="w-6 h-6" />
                                    </div>
                                    <span className="text-lg sm:text-xl font-black text-slate-800 dark:text-white uppercase tracking-widest">Total Income</span>
                                </div>
                                <span className="text-xl sm:text-2xl font-black text-slate-800 dark:text-white tracking-tighter">
                                    <span className="text-sm opacity-40 mr-2">{currency}</span>
                                    {Number(report.income).toLocaleString()}
                                </span>
                            </div>
                            <div className="pl-4 sm:pl-6 space-y-6">
                                <div className="flex flex-col sm:flex-row sm:justify-between sm:items-center gap-2 group">
                                    <span className="text-sm font-bold text-slate-500 group-hover:text-emerald-500 transition-colors uppercase tracking-widest">Revenue</span>
                                    <span className="text-sm sm:text-base font-black text-slate-800 dark:text-white tracking-tighter">
                                        <span className="text-[10px] opacity-40 mr-1">{currency}</span>
                                        {Number(report.income).toLocaleString()}
                                    </span>
                                </div>
                            </div>
                        </div>

                        {/* Expense Section */}
                        <div className="space-y-6 sm:space-y-8">
                            <div className="flex flex-col sm:flex-row sm:justify-between sm:items-center gap-4 pb-4 border-b border-rose-500/20">
                                <div className="flex items-center gap-4">
                                    <div className="p-3 bg-rose-500/10 rounded-xl text-rose-500 shrink-0">
                                        <TrendingDown className="w-6 h-6" />
                                    </div>
                                    <span className="text-lg sm:text-xl font-black text-rose-500 uppercase tracking-widest">Total Expenses</span>
                                </div>
                                <span className="text-xl sm:text-2xl font-black text-rose-500 tracking-tighter">
                                    <span className="text-sm opacity-40 mr-2">{currency}</span>
                                    {Number(report.expenses).toLocaleString()}
                                </span>
                            </div>
                            <div className="pl-4 sm:pl-6 space-y-6">
                                <div className="flex flex-col sm:flex-row sm:justify-between sm:items-center gap-2 group">
                                    <span className="text-sm font-bold text-slate-500 group-hover:text-rose-500 transition-colors uppercase tracking-widest">Operational Expenses</span>
                                    <span className="text-sm sm:text-base font-black text-slate-800 dark:text-white tracking-tighter">
                                        <span className="text-[10px] opacity-40 mr-1">{currency}</span>
                                        {Number(report.expenses).toLocaleString()}
                                    </span>
                                </div>
                            </div>
                        </div>

                        {/* Final Net Section */}
                        <div className="pt-10 border-t-2 border-dashed border-slate-200 dark:border-slate-800">
                            <div className="flex flex-col sm:flex-row sm:justify-between sm:items-center bg-slate-900 dark:bg-black p-6 sm:p-8 rounded-[2rem] sm:rounded-[2.5rem] shadow-2xl relative overflow-hidden group gap-6">
                                <div className={clsx(
                                    "absolute top-0 right-0 w-64 h-64 -mr-32 -mt-32 rounded-full blur-3xl opacity-20 transition-all duration-700",
                                    report.netProfit >= 0 ? "bg-emerald-500" : "bg-rose-500"
                                )} />

                                <div className="relative">
                                    <h3 className="text-[10px] font-black text-slate-400 uppercase tracking-[0.3em] mb-1">Summary</h3>
                                    <p className="text-white/40 text-[9px] font-black uppercase tracking-widest leading-none">Net Result</p>
                                </div>
                                <span className={clsx(
                                    "text-3xl sm:text-4xl font-black tracking-tighter relative",
                                    report.netProfit >= 0 ? "text-emerald-400" : "text-rose-400"
                                )}>
                                    <span className="text-base sm:text-lg opacity-40 mr-2 text-white">{currency}</span>
                                    {Number(report.netProfit).toLocaleString()}
                                </span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    );
};

export default ProfitLossReport;
