import React, { useState, useEffect } from 'react';
import clientPortalService from '../services/clientPortalService';
import {
    History,
    CreditCard,
    ArrowUpRight,
    ShieldCheck,
    Calendar,
    Hash,
    Layers,
    Search
} from 'lucide-react';
import { clsx } from 'clsx';

const ClientPaymentList = () => {
    const [payments, setPayments] = useState([]);
    const [loading, setLoading] = useState(true);
    const [searchTerm, setSearchTerm] = useState('');

    useEffect(() => {
        fetchPayments();
    }, []);

    const fetchPayments = async () => {
        setLoading(true);
        try {
            const res = await clientPortalService.getPayments();
            setPayments(res.data);
        } catch (err) {
            console.error(err);
        } finally {
            setLoading(false);
        }
    };

    const filteredPayments = payments.filter(p =>
        (p.Invoice?.invoiceNumber && p.Invoice.invoiceNumber.toLowerCase().includes(searchTerm.toLowerCase())) ||
        (p.transactionReference && p.transactionReference.toLowerCase().includes(searchTerm.toLowerCase()))
    );

    return (
        <div className="space-y-10 animate-slide-up pb-20">
            {/* Premium Header */}
            <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-6">
                <div className="flex items-center gap-4 sm:gap-5 min-w-0">
                    <div className="p-3 sm:p-4 bg-emerald-500 rounded-2xl text-white shadow-xl shadow-emerald-500/20 shrink-0">
                        <History className="w-6 h-6 sm:w-8 sm:h-8 shrink-0" />
                    </div>
                    <div className="min-w-0">
                        <h1 className="text-xl sm:text-2xl font-extrabold text-slate-800 dark:text-white tracking-tight leading-none truncate">Payment History</h1>
                        <div className="flex items-center gap-2 mt-1">
                            <span className="text-[10px] font-black text-slate-400 dark:text-slate-500 uppercase tracking-widest whitespace-nowrap">All Payments</span>
                            <div className="w-1 h-1 bg-slate-300 dark:bg-slate-700 rounded-full shrink-0"></div>
                            <span className="text-[10px] font-black text-emerald-500 uppercase tracking-widest whitespace-nowrap truncate">{payments.length} Payments</span>
                        </div>
                    </div>
                </div>
                <div className="flex items-center gap-3 px-5 sm:px-6 h-12 bg-emerald-500/5 dark:bg-emerald-500/10 border border-emerald-500/20 rounded-2xl w-full md:w-auto justify-center shrink-0">
                    <ShieldCheck className="w-4 h-4 text-emerald-500 shrink-0" />
                    <span className="text-[10px] font-black text-emerald-600 dark:text-emerald-400 uppercase tracking-widest whitespace-nowrap">End-to-End Encrypted</span>
                </div>
            </div>

            {/* Main Table Card */}
            <div className="premium-card overflow-hidden">
                <div className="p-4 sm:p-6 border-b border-slate-100 dark:border-slate-800/50 flex flex-col md:flex-row justify-between items-center gap-6">
                    <div className="relative flex-1 w-full group">
                        <Search className="absolute left-4 top-1/2 -translate-y-1/2 w-4 h-4 text-slate-300 group-focus-within:text-emerald-500 transition-colors shrink-0" />
                        <input
                            type="text"
                            placeholder="Find payment by invoice or reference..."
                            value={searchTerm}
                            onChange={(e) => setSearchTerm(e.target.value)}
                            className="w-full !pl-12 !py-3 !bg-slate-50/50 dark:!bg-slate-900/40 border-slate-100 dark:border-slate-800/50 outline-none rounded-2xl text-sm"
                        />
                    </div>
                </div>

                <div className="overflow-x-auto overflow-y-hidden">
                    <table className="w-full text-left min-w-[900px]">
                        <thead>
                            <tr className="bg-slate-50/50 dark:bg-slate-900/40 border-b border-slate-100 dark:border-slate-800">
                                <th className="px-6 sm:px-8 py-5 text-[11px] font-black text-slate-400 dark:text-slate-500 uppercase tracking-widest">Date</th>
                                <th className="px-6 sm:px-8 py-5 text-[11px] font-black text-slate-400 dark:text-slate-500 uppercase tracking-widest">Invoice Number</th>
                                <th className="px-6 sm:px-8 py-5 text-[11px] font-black text-slate-400 dark:text-slate-500 uppercase tracking-widest">Payment Method</th>
                                <th className="px-6 sm:px-8 py-5 text-[11px] font-black text-slate-400 dark:text-slate-500 uppercase tracking-widest">Reference</th>
                                <th className="px-6 sm:px-8 py-5 text-[11px] font-black text-slate-400 dark:text-slate-500 uppercase tracking-widest text-right">Amount</th>
                            </tr>
                        </thead>
                        <tbody className="divide-y divide-slate-100 dark:divide-slate-800/50">
                            {loading ? (
                                <tr>
                                    <td colSpan="5" className="p-24 text-center">
                                        <div className="flex flex-col items-center gap-4">
                                            <div className="w-12 h-12 border-4 border-emerald-500/20 border-t-emerald-500 rounded-full animate-spin shrink-0"></div>
                                            <p className="text-[11px] font-black text-slate-400 uppercase tracking-[0.2em]">Loading Payments...</p>
                                        </div>
                                    </td>
                                </tr>
                            ) : filteredPayments.length === 0 ? (
                                <tr>
                                    <td colSpan="5" className="p-12 sm:p-24 text-center">
                                        <div className="max-w-xs mx-auto space-y-4">
                                            <div className="w-16 h-16 bg-slate-50 dark:bg-slate-900 rounded-3xl flex items-center justify-center mx-auto shrink-0">
                                                <CreditCard className="w-8 h-8 text-slate-200 dark:text-slate-800 shrink-0" />
                                            </div>
                                            <h3 className="text-sm font-extrabold text-slate-800 dark:text-white uppercase tracking-tight">No Activity Found</h3>\n                                            <p className="text-[11px] text-slate-500 dark:text-slate-500 leading-relaxed font-bold uppercase tracking-widest">Processed payments will appear here once verified by the system.</p>
                                        </div>
                                    </td>
                                </tr>
                            ) : (
                                filteredPayments.map((p) => (
                                    <tr key={p.id} className="group hover:bg-emerald-50/10 transition-all duration-300">
                                        <td className="px-6 sm:px-8 py-6">
                                            <div className="flex items-center gap-2 text-xs font-bold text-slate-600 dark:text-slate-400 whitespace-nowrap">
                                                <Calendar className="w-3.5 h-3.5 text-slate-300 dark:text-slate-700 shrink-0" />
                                                {new Date(p.paymentDate).toLocaleDateString()}
                                            </div>
                                        </td>
                                        <td className="px-6 sm:px-8 py-6">
                                            <div className="flex items-center gap-2 text-sm font-black text-slate-800 dark:text-white tracking-tight whitespace-nowrap">
                                                <Layers className="w-3.5 h-3.5 text-primary opacity-50 shrink-0" />
                                                #{p.Invoice?.invoiceNumber}
                                            </div>
                                        </td>
                                        <td className="px-6 sm:px-8 py-6">
                                            <span className="px-2.5 py-1 bg-slate-100 dark:bg-slate-800 rounded-lg text-[9px] font-black text-slate-500 dark:text-slate-400 uppercase tracking-widest border border-slate-200 dark:border-slate-700 whitespace-nowrap">
                                                {p.paymentMethod}
                                            </span>
                                        </td>
                                        <td className="px-6 sm:px-8 py-6">
                                            <div className="flex items-center gap-2 text-xs font-mono text-slate-400 dark:text-slate-600 uppercase whitespace-nowrap">
                                                <Hash className="w-3 h-3 opacity-30 shrink-0" />
                                                {p.transactionReference || <span className="opacity-30">Pending...</span>}
                                            </div>
                                        </td>
                                        <td className="px-6 sm:px-8 py-6 text-right">
                                            <div className="flex items-center justify-end gap-1.5 font-black text-emerald-600 tracking-tighter whitespace-nowrap">
                                                <ArrowUpRight className="w-4 h-4 translate-y-px shrink-0" />
                                                <span className="text-[10px] opacity-60 shrink-0">$</span>
                                                <span className="text-sm">{Number(p.amount).toLocaleString()}</span>
                                            </div>
                                        </td>
                                    </tr>
                                ))
                            )}
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    );
};

export default ClientPaymentList;
