import React, { useState, useEffect } from 'react';
import { reportService } from '../services/phase4Services';
import {
    ShieldCheck,
    Download,
    Database,
    Briefcase,
    PieChart,
    ArrowUpRight,
    Search,
    RefreshCw,
    Scale
} from 'lucide-react';
import settingsService from '../services/settingsService';
import { clsx } from 'clsx';

const BalanceSheet = () => {
    const [report, setReport] = useState(null);
    const [loading, setLoading] = useState(true);
    const [currency, setCurrency] = useState('KES');

    useEffect(() => {
        fetchReport();
        settingsService.getSettings().then(res => setCurrency(res.data.DEFAULT_CURRENCY || 'KES'));
    }, []);

    const fetchReport = async () => {
        setLoading(true);
        try {
            const res = await reportService.getBalanceSheet();
            setReport(res.data);
        } catch (err) {
            console.error(err);
        } finally {
            setLoading(false);
        }
    };

    if (loading) return (
        <div className="min-h-[60vh] flex flex-col items-center justify-center gap-4 animate-pulse">
            <div className="w-12 h-12 border-4 border-primary/20 border-t-primary rounded-full animate-spin"></div>
            <p className="text-[10px] font-black text-slate-400 uppercase tracking-[0.2em]">Loading report...</p>
        </div>
    );

    return (
        <div className="space-y-10 animate-slide-up pb-20">
            {/* Premium Header */}
            <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-6">
                <div className="flex items-center gap-4 sm:gap-5 min-w-0">
                    <div className="p-3 sm:p-4 bg-primary rounded-2xl text-white shadow-xl shadow-primary/20 shrink-0">
                        <Scale className="w-6 h-6 sm:w-8 sm:h-8 shrink-0" />
                    </div>
                    <div className="min-w-0">
                        <h1 className="text-xl sm:text-2xl font-extrabold text-slate-800 dark:text-white tracking-tight leading-none truncate">Balance Sheet</h1>
                        <div className="flex items-center gap-2 mt-1">
                            <span className="text-[10px] font-black text-slate-400 dark:text-slate-500 uppercase tracking-widest whitespace-nowrap">Balance Sheet Analysis</span>
                            <div className="w-1 h-1 bg-slate-300 dark:bg-slate-700 rounded-full shrink-0"></div>
                            <span className="text-[10px] font-black text-primary uppercase tracking-widest whitespace-nowrap">Summary</span>
                        </div>
                    </div>
                </div>
                <div className="flex items-center gap-3 w-full md:w-auto">
                    <button
                        onClick={fetchReport}
                        className="btn-secondary h-12 w-12 sm:h-14 sm:w-14 flex items-center justify-center rounded-2xl shrink-0"
                        title="Recalculate Ledger"
                    >
                        <RefreshCw className="w-4 h-4 shrink-0" />
                    </button>
                    <button className="btn-primary flex-1 md:flex-none flex items-center justify-center gap-2 px-8 py-3 shadow-2xl h-12 lg:h-14">
                        <Download className="w-4 h-4 shrink-0" />
                        <span className="text-[10px] sm:text-[11px] font-black uppercase tracking-widest whitespace-nowrap">Export Report</span>
                    </button>
                </div>
            </div>

            <div className="max-w-4xl mx-auto space-y-10">
                {/* Main Balanced Card */}
                <div className="premium-card overflow-hidden">
                    <div className="absolute top-0 right-0 w-64 h-64 -mr-32 -mt-32 bg-primary/5 rounded-full blur-3xl pointer-events-none" />

                    <div className="p-6 sm:p-10 space-y-12 sm:space-y-16 relative">
                        {/* Assets */}
                        <section className="space-y-6 sm:space-y-8">
                            <div className="flex items-center justify-between pb-4 border-b border-slate-900/10 dark:border-white/10 gap-4">
                                <div className="flex items-center gap-3 sm:gap-4">
                                    <div className="p-2.5 sm:p-3 bg-primary/10 rounded-xl text-primary shrink-0">
                                        <Database className="w-5 h-5 sm:w-6 sm:h-6 shrink-0" />
                                    </div>
                                    <h2 className="text-lg sm:text-xl font-black text-slate-800 dark:text-white uppercase tracking-widest truncate">Total Assets</h2>
                                </div>
                                <span className="text-[9px] sm:text-[10px] font-black text-slate-400 uppercase tracking-[0.2em] whitespace-nowrap hidden min-[400px]:block">Summary</span>
                            </div>
                            <div className="space-y-5 sm:space-y-6">
                                <div className="flex justify-between items-center gap-4 group">
                                    <span className="text-[11px] sm:text-sm font-bold text-slate-500 group-hover:text-primary transition-colors uppercase tracking-widest truncate">Cash & Bank</span>
                                    <span className="text-sm font-black text-slate-800 dark:text-white tracking-tighter whitespace-nowrap">
                                        <span className="text-[10px] opacity-40 mr-1 shrink-0">{currency}</span>
                                        {Number(report.assets.cash).toLocaleString()}
                                    </span>
                                </div>
                                <div className="flex justify-between items-center gap-4 group">
                                    <span className="text-[11px] sm:text-sm font-bold text-slate-500 group-hover:text-primary transition-colors uppercase tracking-widest truncate">Accounts Receivable</span>
                                    <span className="text-sm font-black text-slate-800 dark:text-white tracking-tighter whitespace-nowrap">
                                        <span className="text-[10px] opacity-40 mr-1 shrink-0">{currency}</span>
                                        {Number(report.assets.accountsReceivable).toLocaleString()}
                                    </span>
                                </div>
                                <div className="pt-6 border-t border-slate-100 dark:border-slate-800/50 flex flex-col sm:flex-row justify-between items-start sm:items-center gap-2">
                                    <span className="text-sm sm:text-base font-black text-slate-800 dark:text-white uppercase tracking-widest">TOTAL ASSETS</span>
                                    <span className="text-xl sm:text-2xl font-black text-primary tracking-tighter whitespace-nowrap">
                                        <span className="text-sm opacity-40 mr-2 shrink-0">{currency}</span>
                                        {Number(report.assets.total).toLocaleString()}
                                    </span>
                                </div>
                            </div>
                        </section>

                        {/* Liabilities */}
                        <section className="space-y-6 sm:space-y-8">
                            <div className="flex items-center justify-between pb-4 border-b border-rose-500/20 gap-4">
                                <div className="flex items-center gap-3 sm:gap-4">
                                    <div className="p-2.5 sm:p-3 bg-rose-500/10 rounded-xl text-rose-500 shrink-0">
                                        <Briefcase className="w-5 h-5 sm:w-6 sm:h-6 shrink-0" />
                                    </div>
                                    <h2 className="text-lg sm:text-xl font-black text-rose-500 uppercase tracking-widest truncate">Total Liabilities</h2>
                                </div>
                                <span className="text-[9px] sm:text-[10px] font-black text-rose-500/40 uppercase tracking-[0.2em] whitespace-nowrap hidden min-[400px]:block">Summary</span>
                            </div>
                            <div className="space-y-5 sm:space-y-6">
                                <div className="flex justify-between items-center gap-4 group">
                                    <span className="text-[11px] sm:text-sm font-bold text-slate-500 group-hover:text-rose-500 transition-colors uppercase tracking-widest truncate">Accounts Payable</span>
                                    <span className="text-sm font-black text-slate-800 dark:text-white tracking-tighter whitespace-nowrap">
                                        <span className="text-[10px] opacity-40 mr-1 shrink-0">{currency}</span>
                                        0.00
                                    </span>
                                </div>
                                <div className="pt-6 border-t border-rose-100 dark:border-rose-900/30 flex flex-col sm:flex-row justify-between items-start sm:items-center gap-2">
                                    <span className="text-sm sm:text-base font-black text-rose-500 tracking-widest uppercase">TOTAL LIABILITIES</span>
                                    <span className="text-xl sm:text-2xl font-black text-rose-500 tracking-tighter whitespace-nowrap">
                                        <span className="text-sm opacity-40 mr-2 shrink-0">{currency}</span>
                                        0.00
                                    </span>
                                </div>
                            </div>
                        </section>

                        {/* Equity */}
                        <section className="p-6 sm:p-10 bg-indigo-500/5 dark:bg-indigo-500/10 rounded-[2rem] sm:rounded-[2.5rem] border border-indigo-500/10 space-y-6 sm:space-y-8">
                            <div className="flex items-center justify-between pb-4 border-b border-indigo-500/20 gap-4">
                                <div className="flex items-center gap-3 sm:gap-4">
                                    <div className="p-2.5 sm:p-3 bg-indigo-500/10 rounded-xl text-indigo-500 shrink-0">
                                        <PieChart className="w-5 h-5 sm:w-6 sm:h-6 shrink-0" />
                                    </div>
                                    <h2 className="text-lg sm:text-xl font-black text-indigo-500 uppercase tracking-widest truncate">Shareholder Equity</h2>
                                </div>
                                <span className="text-[9px] sm:text-[10px] font-black text-indigo-500/40 uppercase tracking-[0.2em] whitespace-nowrap hidden min-[400px]:block">Balance Sheet</span>
                            </div>
                            <div className="space-y-5 sm:space-y-6">
                                <div className="flex justify-between items-center gap-4 group">
                                    <span className="text-[11px] sm:text-sm font-bold text-slate-500 group-hover:text-indigo-500 transition-colors uppercase tracking-widest truncate">Retained Earnings</span>
                                    <span className="text-sm font-black text-slate-800 dark:text-white tracking-tighter whitespace-nowrap">
                                        <span className="text-[10px] opacity-40 mr-1 shrink-0">{currency}</span>
                                        {Number(report.equity.retainedEarnings).toLocaleString()}
                                    </span>
                                </div>
                                <div className="pt-6 border-t border-indigo-200 dark:border-indigo-900/50 flex flex-col sm:flex-row justify-between items-start sm:items-center gap-2">
                                    <span className="text-sm sm:text-base font-black text-indigo-700 dark:text-indigo-400 uppercase tracking-widest">TOTAL EQUITY</span>
                                    <span className="text-xl sm:text-2xl font-black text-indigo-600 dark:text-indigo-300 tracking-tighter whitespace-nowrap">
                                        <span className="text-sm opacity-40 mr-2 shrink-0">{currency}</span>
                                        {Number(report.equity.total).toLocaleString()}
                                    </span>
                                </div>
                            </div>
                        </section>

                        <div className="pt-8 flex flex-col items-center gap-4">
                            <div className="flex items-center gap-3 px-6 py-3 bg-slate-50 dark:bg-slate-900/60 rounded-full border border-slate-100 dark:border-slate-800 animate-pulse text-center">
                                <ShieldCheck className="w-5 h-5 text-emerald-500 shrink-0" />
                                <span className="text-[9px] sm:text-[10px] font-black text-slate-500 dark:text-slate-400 uppercase tracking-[0.2em]">Verified: Assets = Liabilities + Equity</span>
                            </div>
                            <p className="text-[9px] font-black text-slate-300 dark:text-slate-700 uppercase tracking-widest">Account balances verified.</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    );
};

export default BalanceSheet;
