import React from 'react';
import { useAuth } from '../contexts/AuthContext';
import { useTheme } from '../contexts/ThemeContext';
import { Search, Bell, LogOut, Sun, Moon, User, Menu } from 'lucide-react';

const Topbar = ({ toggleSidebar }) => {
    const { user, logout } = useAuth();
    const { darkMode, toggleDarkMode } = useTheme();

    const getGreeting = () => {
        const hour = new Date().getHours();
        if (hour < 12) return "Good morning ☀️";
        if (hour < 18) return "Good afternoon 🌤️";
        return "Good evening 🌙";
    };

    return (
        <header className="h-16 sm:h-24 bg-transparent flex items-center justify-between px-4 sm:px-8 sticky top-0 z-30 transition-all duration-300">
            <div className="flex items-center gap-6 flex-1">
                <button
                    onClick={toggleSidebar}
                    className="lg:hidden p-2 text-slate-400 hover:text-primary transition-colors"
                >
                    <Menu className="w-6 h-6" />
                </button>

                <div className="hidden lg:block shrink-0">
                    <h2 className="text-2xl font-black text-slate-800 dark:text-white flex items-center gap-2 tracking-tight">
                        {user?.fullName?.split(' ')[0] || 'Member'}
                    </h2>
                </div>

                <div className="hidden md:flex relative max-w-md w-full ml-8 group">
                    <Search className="absolute left-5 top-1/2 -translate-y-1/2 w-5 h-5 text-slate-400 group-focus-within:text-primary transition-colors" />
                    <input
                        type="text"
                        placeholder="Search a transaction..."
                        className="w-full pl-14 pr-6 py-4 bg-white dark:bg-bg-darkCard border-none ring-1 ring-slate-100 dark:ring-slate-800 rounded-[20px] text-sm font-medium text-slate-600 dark:text-slate-200 focus:outline-none focus:ring-2 focus:ring-primary/50 shadow-sm transition-all placeholder:text-slate-400"
                    />
                </div>
            </div>

            <div className="flex items-center gap-4 sm:gap-6 ml-4">
                <div className="flex items-center gap-2">
                    <button
                        onClick={toggleDarkMode}
                        className="w-12 h-12 flex items-center justify-center bg-white dark:bg-bg-darkCard hover:bg-slate-50 dark:hover:bg-slate-800 rounded-[18px] transition-all text-slate-400 hover:text-primary ring-1 ring-slate-100 dark:ring-slate-800 shadow-sm"
                        title={darkMode ? 'Switch to Light Mode' : 'Switch to Dark Mode'}
                    >
                        {darkMode ? <Sun className="w-5 h-5" /> : <Moon className="w-5 h-5" />}
                    </button>

                    <button className="w-12 h-12 flex items-center justify-center bg-white dark:bg-bg-darkCard hover:bg-slate-50 dark:hover:bg-slate-800 rounded-[18px] transition-all text-slate-400 hover:text-primary ring-1 ring-slate-100 dark:ring-slate-800 shadow-sm relative group">
                        <Bell className="w-5 h-5 group-hover:animate-swing" />
                        <span className="absolute top-3 right-3 w-2.5 h-2.5 bg-rose-500 rounded-full border-2 border-white dark:border-bg-darkCard animate-pulse"></span>
                    </button>
                </div>

                <div className="pl-4 border-l border-slate-200 dark:border-slate-800">
                    <div className="flex items-center gap-3 group cursor-pointer relative">
                        <div className="w-12 h-12 rounded-[18px] bg-gradient-to-tr from-primary to-blue-600 flex items-center justify-center text-white font-black shadow-lg shadow-primary/30 transition-transform group-hover:scale-105 active:scale-95 overflow-hidden ring-4 ring-white dark:ring-bg-dark">
                            {user?.avatar ? (
                                <img src={user.avatar} alt="User" className="w-full h-full object-cover" />
                            ) : (
                                <span>{user?.fullName?.charAt(0) || 'U'}</span>
                            )}
                        </div>
                        <div className="hidden xl:block">
                            <p className="text-sm font-black text-slate-800 dark:text-white leading-tight">{user?.fullName || 'Guest User'}</p>
                            <div className="flex items-center gap-1 mt-0.5">
                                <span className="w-1.5 h-1.5 rounded-full bg-emerald-500"></span>
                                <p className="text-[10px] font-bold text-slate-400 uppercase tracking-widest">{user?.role || 'VIEWER'}</p>
                            </div>
                        </div>

                        {/* Profile Dropdown */}
                        <div className="absolute right-0 top-full mt-4 w-64 bg-white dark:bg-bg-darkCard border border-slate-100 dark:border-slate-800 rounded-[24px] shadow-premium-deep opacity-0 group-hover:opacity-100 invisible group-hover:visible transition-all duration-300 translate-y-2 group-hover:translate-y-0 z-50 overflow-hidden transform origin-top-right">
                            <div className="p-2">
                                <div className="p-4 bg-slate-50 dark:bg-slate-800/50 rounded-2xl mb-2 flex items-center gap-4">
                                    <div className="w-10 h-10 rounded-xl bg-primary/10 flex items-center justify-center text-primary font-bold">
                                        {user?.fullName?.charAt(0) || 'U'}
                                    </div>
                                    <div>
                                        <p className="text-sm font-bold text-slate-800 dark:text-white">{user?.fullName}</p>
                                        <p className="text-[10px] text-slate-500 font-bold break-all">{user?.email}</p>
                                    </div>
                                </div>
                                <div className="space-y-1">
                                    <button className="w-full flex items-center gap-3 px-4 py-3 text-xs font-bold uppercase tracking-wider text-slate-500 dark:text-slate-400 hover:bg-slate-50 dark:hover:bg-slate-800 hover:text-primary rounded-xl transition-all">
                                        <User className="w-4 h-4" /> My Account
                                    </button>
                                    <button
                                        onClick={logout}
                                        className="w-full flex items-center gap-3 px-4 py-3 text-xs font-bold uppercase tracking-wider text-rose-500 hover:bg-rose-50 dark:hover:bg-rose-500/10 rounded-xl transition-all"
                                    >
                                        <LogOut className="w-4 h-4" /> Sign Out
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </header>
    );
};

export default Topbar;
