import React from 'react';
import { BrowserRouter as Router, Routes, Route, Navigate } from 'react-router-dom';
import { AuthProvider, useAuth } from './contexts/AuthContext';
import { ThemeProvider } from './contexts/ThemeContext';
import MainLayout from './layouts/MainLayout';
import Dashboard from './pages/Dashboard';
import Login from './pages/Login';
import IncomeList from './pages/IncomeList';
import ExpenseList from './pages/ExpenseList';
import VendorList from './pages/VendorList';
import CategoryList from './pages/CategoryList';
import IncomeCategoryList from './pages/IncomeCategoryList';
import InvoiceList from './pages/InvoiceList';
import QuoteList from './pages/QuoteList';
import ClientList from './pages/ClientList';
import SubscriptionList from './pages/SubscriptionList';
import ProfitLossReport from './pages/ProfitLossReport';
import BalanceSheet from './pages/BalanceSheet';
import ClientDashboard from './pages/ClientDashboard';
import ClientInvoiceList from './pages/ClientInvoiceList';
import ClientQuoteList from './pages/ClientQuoteList';
import ClientPaymentList from './pages/ClientPaymentList';
import ClientProfile from './pages/ClientProfile';
import Settings from './pages/Settings';
import UserList from './pages/UserList';
import PayrollDashboard from './pages/PayrollDashboard';
import BudgetDashboard from './pages/BudgetDashboard';
import StaffList from './pages/StaffList';
import StaffPayments from './pages/StaffPayments';
import CrmAutomation from './pages/CrmAutomation';
import Banking from './pages/Banking';

// --- Guards ---
const ProtectedRoute = ({ children }) => {
    const { user, loading } = useAuth();

    if (loading) return (
        <div className="min-h-screen flex items-center justify-center bg-slate-50 dark:bg-slate-900 transition-colors duration-300">
            <div className="w-12 h-12 border-4 border-blue-600 border-t-transparent rounded-full animate-spin"></div>
        </div>
    );

    if (!user) return <Navigate to="/login" replace />;

    return <MainLayout>{children}</MainLayout>;
};

const PublicRoute = ({ children }) => {
    const { user, loading } = useAuth();
    if (loading) return null;
    if (user) return <Navigate to="/dashboard" replace />;
    return children;
};

// --- Module Placeholder Component ---
const PlaceholderPage = ({ title }) => (
    <div className="flex flex-col items-center justify-center min-h-[60vh] text-center">
        <div className="w-20 h-20 bg-blue-50 dark:bg-blue-900/20 text-blue-600 rounded-2xl flex items-center justify-center mb-6">
            <h2 className="text-3xl font-bold">TODO</h2>
        </div>
        <h1 className="text-2xl font-bold text-slate-800 dark:text-white">{title} Module</h1>
        <p className="text-slate-500 mt-2 max-w-md">
            This module is scheduled for development in the next phase of the implementation plan.
        </p>
    </div>
);

function App() {
    return (
        <ThemeProvider>
            <AuthProvider>
                <Router>
                    <Routes>
                        {/* Public Auth Routes */}
                        <Route path="/login" element={<PublicRoute><Login /></PublicRoute>} />

                        {/* Protected Dashboard & Modules */}
                        <Route path="/dashboard" element={<ProtectedRoute><Dashboard /></ProtectedRoute>} />

                        {/* Module Placeholders */}
                        <Route path="/income" element={<ProtectedRoute><IncomeList /></ProtectedRoute>} />
                        <Route path="/expenses" element={<ProtectedRoute><ExpenseList /></ProtectedRoute>} />
                        <Route path="/vendors" element={<ProtectedRoute><VendorList /></ProtectedRoute>} />
                        <Route path="/categories" element={<ProtectedRoute><CategoryList /></ProtectedRoute>} />
                        <Route path="/income-categories" element={<ProtectedRoute><IncomeCategoryList /></ProtectedRoute>} />
                        <Route path="/invoices" element={<ProtectedRoute><InvoiceList /></ProtectedRoute>} />
                        <Route path="/quotes" element={<ProtectedRoute><QuoteList /></ProtectedRoute>} />
                        <Route path="/clients" element={<ProtectedRoute><ClientList /></ProtectedRoute>} />
                        <Route path="/payroll" element={<ProtectedRoute><PayrollDashboard /></ProtectedRoute>} />
                        <Route path="/subscriptions" element={<ProtectedRoute><SubscriptionList /></ProtectedRoute>} />
                        <Route path="/budgets" element={<ProtectedRoute><BudgetDashboard /></ProtectedRoute>} />
                        <Route path="/staff" element={<ProtectedRoute><StaffList /></ProtectedRoute>} />
                        <Route path="/staff-payments" element={<ProtectedRoute><StaffPayments /></ProtectedRoute>} />
                        <Route path="/banking" element={<ProtectedRoute><Banking /></ProtectedRoute>} />
                        <Route path="/crm-automation" element={<ProtectedRoute><CrmAutomation /></ProtectedRoute>} />
                        <Route path="/reports/profit-loss" element={<ProtectedRoute><ProfitLossReport /></ProtectedRoute>} />
                        <Route path="/reports/balance-sheet" element={<ProtectedRoute><BalanceSheet /></ProtectedRoute>} />

                        {/* Client Portal Routes */}
                        <Route path="/portal/dashboard" element={<ProtectedRoute><ClientDashboard /></ProtectedRoute>} />
                        <Route path="/portal/invoices" element={<ProtectedRoute><ClientInvoiceList /></ProtectedRoute>} />
                        <Route path="/portal/quotes" element={<ProtectedRoute><ClientQuoteList /></ProtectedRoute>} />
                        <Route path="/portal/payments" element={<ProtectedRoute><ClientPaymentList /></ProtectedRoute>} />
                        <Route path="/portal/profile" element={<ProtectedRoute><ClientProfile /></ProtectedRoute>} />

                        <Route path="/settings" element={<ProtectedRoute><Settings /></ProtectedRoute>} />
                        <Route path="/users" element={<ProtectedRoute><UserList /></ProtectedRoute>} />

                        {/* Root Redirect */}
                        <Route path="/" element={<Navigate to="/dashboard" replace />} />
                    </Routes>
                </Router>
            </AuthProvider>
        </ThemeProvider>
    );
}

export default App;