const { sequelize } = require('./config/db');
const { User } = require('./models/foundation');
const { ExpenseCategory, Vendor, Expense } = require('./models/expense');

const seed = async () => {
    try {
        await sequelize.sync({ force: false });

        // 1. Get Admin User
        const adminUser = await User.findOne({ where: { email: 'admin@financepro.com' } });
        if (!adminUser) {
            console.error('❌ Admin user not found. Please run seed_phase1.js first.');
            process.exit(1);
        }

        // 2. Expense Categories
        const categories = [
            { name: 'Rent & Office', description: 'Office lease and management fees' },
            { name: 'Utilities', description: 'Electricity, Water, Trash' },
            { name: 'SaaS & Software', description: 'Tools like AWS, Github, Slack' },
            { name: 'Salaries & Wages', description: 'Employee payroll' },
            { name: 'Marketing', description: 'Ads and lead generation' },
            { name: 'Travel & Meals', description: 'Business travel costs' },
        ];

        for (const cat of categories) {
            await ExpenseCategory.findOrCreate({ where: { name: cat.name }, defaults: cat });
        }

        // 3. Vendors
        const vendors = [
            { name: 'Amazon Web Services', email: 'billing@aws.com', notes: 'Main hosting provider' },
            { name: 'Google Cloud', email: 'billing@google.com' },
            { name: 'Shared Office Co.', contactPerson: 'John Smith', phone: '+123456789' },
            { name: 'Digital Ocean', email: 'billing@digitalocean.com' },
        ];

        for (const v of vendors) {
            await Vendor.findOrCreate({ where: { name: v.name }, defaults: v });
        }

        const saasCat = await ExpenseCategory.findOne({ where: { name: 'SaaS & Software' } });
        const rentCat = await ExpenseCategory.findOne({ where: { name: 'Rent & Office' } });
        const awsVendor = await Vendor.findOne({ where: { name: 'Amazon Web Services' } });
        const officeVendor = await Vendor.findOne({ where: { name: 'Shared Office Co.' } });

        // 4. Sample Expenses
        const expenseData = [
            {
                amount: 450.00,
                expenseCategoryId: saasCat.id,
                vendorId: awsVendor.id,
                paymentMethod: 'Credit Card',
                reference: 'AWS-99211',
                expenseDate: '2026-01-15',
                createdBy: adminUser.id,
                notes: 'Monthly infrastructure bill'
            },
            {
                amount: 2500.00,
                expenseCategoryId: rentCat.id,
                vendorId: officeVendor.id,
                paymentMethod: 'Bank Transfer',
                reference: 'RENT-JAN-26',
                expenseDate: '2026-01-01',
                createdBy: adminUser.id,
                notes: 'January office rent'
            },
            {
                amount: 85.00,
                expenseCategoryId: saasCat.id,
                vendorId: null,
                paymentMethod: 'Credit Card',
                reference: 'SLACK-9112',
                expenseDate: '2026-01-18',
                createdBy: adminUser.id,
                notes: 'Slack subscription'
            }
        ];

        for (const data of expenseData) {
            const exists = await Expense.findOne({ where: { reference: data.reference } });
            if (!exists) await Expense.create(data);
        }

        console.log('✅ Phase 2 Seeding Complete');
        process.exit(0);
    } catch (err) {
        console.error('❌ Seeding failed:', err);
        process.exit(1);
    }
};

seed();
