const { DataTypes } = require('sequelize');
const { sequelize } = require('../config/db');

// --- Role Model ---
const Role = sequelize.define('Role', {
    name: { type: DataTypes.STRING, unique: true, allowNull: false },
    constant: { type: DataTypes.STRING, unique: true, allowNull: false } // e.g., 'SUPER_ADMIN'
}, { timestamps: false });

// --- Permission Model ---
const Permission = sequelize.define('Permission', {
    name: { type: DataTypes.STRING, unique: true, allowNull: false },
    module: { type: DataTypes.STRING, allowNull: false } // e.g., 'INCOME'
}, { timestamps: false });

// --- User Model ---
const User = sequelize.define('User', {
    id: { type: DataTypes.UUID, defaultValue: DataTypes.UUIDV4, primaryKey: true },
    fullName: { type: DataTypes.STRING, allowNull: false },
    email: { type: DataTypes.STRING, unique: true, allowNull: false },
    phone: { type: DataTypes.STRING, unique: true },
    password: { type: DataTypes.STRING, allowNull: false },
    isActive: { type: DataTypes.BOOLEAN, defaultValue: true },
    mustChangePassword: { type: DataTypes.BOOLEAN, defaultValue: false },
    clientId: { type: DataTypes.UUID }, // For CLIENT role users
    credentialsSent: { type: DataTypes.BOOLEAN, defaultValue: false },
    resetToken: { type: DataTypes.STRING },
    tokenExpiry: { type: DataTypes.DATE },
    otp: { type: DataTypes.STRING },
    otpExpiry: { type: DataTypes.DATE },
    lastLogin: { type: DataTypes.DATE }
}, { timestamps: true });

// --- Audit Log Model ---
const AuditLog = sequelize.define('AuditLog', {
    userId: { type: DataTypes.UUID },
    action: { type: DataTypes.STRING, allowNull: false },
    module: { type: DataTypes.STRING, allowNull: false },
    details: { type: DataTypes.TEXT },
    ipAddress: { type: DataTypes.STRING }
}, { timestamps: true, updatedAt: false });

// --- Email Log Model ---
const EmailLog = sequelize.define('EmailLog', {
    to: { type: DataTypes.STRING, allowNull: false },
    subject: { type: DataTypes.STRING },
    module: { type: DataTypes.STRING }, // e.g., 'INVOICE'
    referenceId: { type: DataTypes.STRING }, // e.g., Invoice ID
    status: { type: DataTypes.STRING }, // 'Success', 'Failed'
    errorMessage: { type: DataTypes.TEXT }
}, { timestamps: true, updatedAt: false });

// --- Setting Model ---
const Setting = sequelize.define('Setting', {
    key: { type: DataTypes.STRING, unique: true, allowNull: false },
    value: { type: DataTypes.TEXT },
    type: { type: DataTypes.STRING, defaultValue: 'string' }
}, { timestamps: false });

const { Staff } = require('./staff');
const { StaffPayment } = require('./staffPayment');

// Relations
Role.hasMany(User, { foreignKey: 'roleId' });
User.belongsTo(Role, { foreignKey: 'roleId' });

Role.belongsToMany(Permission, { through: 'RolePermissions', timestamps: false });
Permission.belongsToMany(Role, { through: 'RolePermissions', timestamps: false });

Staff.hasMany(StaffPayment, { foreignKey: 'staffId' });
StaffPayment.belongsTo(Staff, { foreignKey: 'staffId' });

module.exports = { User, Role, Permission, AuditLog, EmailLog, Setting, Staff, StaffPayment };
