const expenseService = require('../services/expenseService');
const pdfService = require('../services/pdfService');

exports.createExpense = async (req, res) => {
    try {
        const expenseData = req.body;
        if (req.file) {
            expenseData.receiptPath = req.file.filename;
        }
        const expense = await expenseService.createExpense(expenseData, req.user.id);
        res.status(201).json(expense);
    } catch (err) {
        res.status(400).json({ error: err.message });
    }
};

exports.getExpenses = async (req, res) => {
    try {
        const expenses = await expenseService.getAllExpenses(req.query);
        res.json(expenses);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

exports.getExpense = async (req, res) => {
    try {
        const expense = await expenseService.getExpenseById(req.params.id);
        if (!expense) return res.status(404).json({ message: 'Not found' });
        res.json(expense);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

exports.updateExpense = async (req, res) => {
    try {
        const updateData = req.body;
        if (req.file) {
            updateData.receiptPath = req.file.filename;
        }
        const expense = await expenseService.updateExpense(req.params.id, updateData, req.user.id);
        res.json(expense);
    } catch (err) {
        res.status(400).json({ error: err.message });
    }
};

exports.deleteExpense = async (req, res) => {
    try {
        await expenseService.deleteExpense(req.params.id, req.user.id);
        res.status(204).send();
    } catch (err) {
        res.status(400).json({ error: err.message });
    }
};

exports.getSummary = async (req, res) => {
    try {
        const summary = await expenseService.getExpenseSummary();
        res.json(summary);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

// --- Meta Routes ---
exports.getCategories = async (req, res) => {
    try {
        const cats = await expenseService.getCategories();
        res.json(cats);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

exports.createCategory = async (req, res) => {
    try {
        const cat = await expenseService.createCategory(req.body);
        res.status(201).json(cat);
    } catch (err) {
        res.status(400).json({ error: err.message });
    }
};

exports.updateCategory = async (req, res) => {
    try {
        const cat = await expenseService.updateCategory(req.params.id, req.body);
        res.json(cat);
    } catch (err) {
        res.status(400).json({ error: err.message });
    }
};

exports.deleteCategory = async (req, res) => {
    try {
        await expenseService.deleteCategory(req.params.id);
        res.json({ message: 'Category deleted' });
    } catch (err) {
        res.status(400).json({ error: err.message });
    }
};

exports.getVendors = async (req, res) => {
    try {
        const vendors = await expenseService.getAllVendors();
        res.json(vendors);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};

exports.createVendor = async (req, res) => {
    try {
        const vendor = await expenseService.createVendor(req.body, req.user.id);
        res.status(201).json(vendor);
    } catch (err) {
        res.status(400).json({ error: err.message });
    }
};

exports.updateVendor = async (req, res) => {
    try {
        const vendor = await expenseService.updateVendor(req.params.id, req.body, req.user.id);
        res.json(vendor);
    } catch (err) {
        res.status(400).json({ error: err.message });
    }
};

exports.deleteVendor = async (req, res) => {
    try {
        await expenseService.deleteVendor(req.params.id, req.user.id);
        res.json({ message: 'Vendor deleted' });
    } catch (err) {
        console.error('Delete Vendor Error:', err);
        res.status(400).json({ error: err.message });
    }
};
exports.downloadPDF = async (req, res) => {
    try {
        const expense = await expenseService.getExpenseById(req.params.id);
        if (!expense) return res.status(404).json({ message: 'Expense not found' });

        res.setHeader('Content-Type', 'application/pdf');
        res.setHeader('Content-Disposition', `attachment; filename=expense-${expense.id.split('-')[0]}.pdf`);

        await pdfService.generateExpensePDF(expense, res);
    } catch (err) {
        res.status(500).json({ error: err.message });
    }
};
